
/* Copyright (C) 2001-2007 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_canon.c */

#include "fs_itype.h"

#ifdef FS_ACT3

static int get_next_token(_DS_ MTX_RA_DECOMP *r);

/****************************************************************/
/****************************************************************/
/* given a RAC3 file in memory, read in all the necessary glop
* to facilitate decoding a file
*
* code assumes that a BYTE pointer can be any address
* if NOT, you will need to allocate and read in the BYTES
* Also you'll need to free them in MTX_RA_DECOMP_Destroy
*
*/

MTX_RA_DECOMP *MTX_RA_DECOMP_Create(_DS_ FS_BYTE *data)
{
    BITIO *bio;
    MTX_RA_DECOMP *r;
    int offset_length;    /* length in bytes of each offset entry. 0 => inferred offsets */
    FS_ULONG *p;
    FS_LONG i, n, offset, crumbs;
    FS_LONG fsize;        /*** bytes in the compressed file ... ***/

    /* verify 'RAC3' signature */
    if (data[0] != 'R' || data[1] != 'A' || data[2] != 'C' || data[3] != 3)
    {
        return 0;
    }

    /* get compressed file size */
    fsize = data[4];
    fsize = (fsize << 8) | data[5];
    fsize = (fsize << 8) | data[6];
    fsize = (fsize << 8) | data[7];

    /* create BITIO and skip signature and compressed file size */
#ifdef FS_MEM_DBG
    STATE.memdbgid = "MTX_RA_DECOMP";
#endif
    r = (MTX_RA_DECOMP*) FSS_calloc(_PS_ sizeof(MTX_RA_DECOMP));
    if (r == 0)
        return 0;

    bio = MTX_BITIO_Create( _PS_ data, fsize);
    if (bio == 0)
    {
        MTX_RA_DECOMP_Destroy(_PS_ r);
        return NULL;
    }

    MTX_BITIO_fseek(bio, 8 << 3);
    r->bio = bio;

    /* Initialize the Huffman code lookup table */
    for (i = 0; i < TABLE_SIZE; i++)
        r->token_bits_table[i] = 0;  /* means no entry for code i */
    r->token_num_table[0] = -1;      /* means table not built     */

    /* read uncompressed file size */
    r->unCompressedByteCount = MTX_BITIO_Read32(bio);

    /* read number of tokens */
    r->num_tokens = MTX_BITIO_Read32(bio);

    /* establish pointer to lengths[] -- and skip the data */
    offset = MTX_BITIO_ftell(bio);
    r->lengths = data + (offset >> 3);
    offset += (r->num_tokens << 3);
    MTX_BITIO_fseek(bio, offset);

    /* read in the offset_length */
    offset_length = MTX_BITIO_Read8(bio);

    /* allocate and calculate or read the offsets */
    n = r->num_tokens;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "r->offsets";
#endif
    p = r->offsets = (FS_ULONG *)FSS_malloc(_PS_ n * 4);
    if (p == 0)
    {
        MTX_RA_DECOMP_Destroy(_PS_ r);
        return NULL;
    }

    if (offset_length == 0)
    {
        /* infer the offsets of the non_overlapping token strings from their lengths */
        p[0] = 0;
        for (i = 1; i < r->num_tokens; i++)
            p[i] = p[i - 1] + r->lengths[i - 1];
    }
    else
    {
        /* read in the offsets of the overlapping token strings */
        offset_length <<= 3;
        for (i = 0; i < r->num_tokens; i++)
            p[i] = MTX_BITIO_ReadValue((BITIO *)(r->bio), offset_length);
    }

    /* read total string data length */
    r->string_len = MTX_BITIO_Read32(bio);

    /* establish pointer to string[] -- and skip the data */
    offset = MTX_BITIO_ftell(bio);
    r->string = data + (offset >> 3);
    offset += (r->string_len << 3);
    MTX_BITIO_fseek(bio, offset);

    /* read number of canonical huffman code groups */
    r->num_groups = MTX_BITIO_Read8(bio);

    /* establish pointer to bits[] -- and skip the data */
    offset = MTX_BITIO_ftell(bio);
    r->bits = data + (offset >> 3);
    offset += (r->num_groups << 3);
    MTX_BITIO_fseek(bio, offset);

    /* allocate and read in firsts[]  */
    n = r->num_groups;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "r->firsts";
#endif
    p = r->firsts = (FS_ULONG *)FSS_malloc(_PS_ n * 4);
    if (p == 0)
    {
        MTX_RA_DECOMP_Destroy(_PS_ r);
        return NULL;
    }

    while (n--)
        *p++ = MTX_BITIO_Read32(bio);

    /* allocate and read in lasts[]  */
    n = r->num_groups;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "r->lasts";
#endif
    p = r->lasts = (FS_ULONG *)FSS_malloc(_PS_ n * 4);
    if (p == 0)
    {
        MTX_RA_DECOMP_Destroy(_PS_ r);
        return NULL;
    }

    while (n--)
        *p++ = MTX_BITIO_Read32(bio);

    /* calculate next_mask to keep rightmost MAX bits */
    if (r->bits[r->num_groups - 1] == 32)
        r->next_mask = 0xFFFFFFFF;
    else
        r->next_mask = (1 << r->bits[r->num_groups - 1]) - 1;

    /* allocate and calculate tokens[]  */
    n = r->num_groups;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "r->tokens";
#endif
    p = r->tokens = (FS_ULONG *)FSS_malloc(_PS_ n * 4);
    if (p == 0)
    {
        MTX_RA_DECOMP_Destroy(_PS_ r);
        return NULL;
    }

    p[0] = 0;
    for (i = 1; i < n; i++)
        p[i] = p[i - 1] + 1 + (r->lasts[i - 1] - r->firsts[i - 1]);

    /* read crumbs (size of bitMarks and deltaBytes arrays) */
    crumbs = MTX_BITIO_Read32(bio);

    /*  bitMarks[]  */
    r->bitMarks = &bio->array[bio->index >> 3]; /* point at the ROM, we'll decode the LONGs on the fly */
    bio->index += 32 * crumbs;

    /* establish pointer to deltaBytes[] */
    n = crumbs;
    offset = MTX_BITIO_ftell(bio);
    r->deltaBytes = data + (offset >> 3);
    offset += n << 3;
    MTX_BITIO_fseek(bio, offset);

    /* read token_offset */
    r->token_offset = MTX_BITIO_Read32(bio);

    /* Convert pointers to offsets for multi-processing */
    return r;
}

/****************************************************************/

FS_VOID MTX_RA_DECOMP_Destroy(_DS_ MTX_RA_DECOMP *t)
{
    if (t)
    {
        FSS_free(_PS_ (FS_ULONG *)(t->firsts));
        FSS_free(_PS_ (FS_ULONG *)(t->lasts));
        FSS_free(_PS_ (FS_ULONG *)(t->tokens));
        FSS_free(_PS_ (FS_ULONG *)(t->offsets));
        FSS_free(_PS_ (BITIO *)(t->bio));
        FSS_free(_PS_ t);
    }
}

/****************************************************************/

/* from current position (assuming someone else has done the
* MTX_BITIO_fseek() and read in <next>) get the next token
* number and refresh <next>.
*/

/****************************************************************/
static FS_VOID build_token_table(MTX_RA_DECOMP *r)
{
    int i, j, k, bits, count, num, token, index;

    for (i = 0, index = 0; ((index < TABLE_SIZE) && (i < r->num_groups) && (r->bits[i] <= TABLE_BITS)); i++)
    {
        bits = r->bits[i];
        count = 1 << (TABLE_BITS - bits);
        num = 1 + r->lasts[i] - r->firsts[i];
        for (j = 0, token = r->tokens[i]; (index < TABLE_SIZE) &&  (j < num); j++, token++)
        {
            for (k = 0; (index < TABLE_SIZE) &&  (k < count); k++)
            {
                r->token_num_table[index] = token;
                r->token_bits_table[index] = (FS_BYTE)bits;
                index++;
            }
        }
    }
    r->slowGroup = i;   /* first group not completely contained in the table */
}

/****************************************************************/
static int get_next_token(_DS_ MTX_RA_DECOMP *r)
{
    int lo, hi, mid;
    FS_ULONG first;
    FS_ULONG last;
    FS_ULONG next;
    FS_ULONG rest;
    int bitsNeeded;
    int drop;
    FS_LONG token_num;
    FS_BYTE n;
    FS_BYTE max;

    /* do we need to build the token table ? */
    if (r->token_num_table[0] < 0)
        build_token_table(r);

    next = r->next;

    /* ? possible && legitimate entry in the table */
    if (r->token_bits_table[next])
    {
        n = r->token_bits_table[next];
        token_num = r->token_num_table[next];
        next <<= n;
        next |= MTX_BITIO_ReadValue(r->bio, n);
        next &= NEXT_MASK;
        r->next = next;
        return token_num;
    }
    else
    {
        /* not in table ... binary search the longer length groups */
        max = r->bits[r->num_groups - 1];
        bitsNeeded = max - TABLE_BITS;
        if (bitsNeeded > 0)
            r->next = (next << bitsNeeded) | MTX_BITIO_ReadValue(r->bio, bitsNeeded);
        else if (bitsNeeded < 0)
        {
            int b = -bitsNeeded;
            r->next >>= b;
            r->bio->index += bitsNeeded;
        }
        lo = r->slowGroup;
        hi = r->num_groups - 1;
        while (lo <= hi)
        {
            mid = (lo + hi) / 2;
            n = r->bits[mid];
            drop = max - n;
            next = r->next >> drop;
            first = r->firsts[mid];
            last  = r->lasts[mid];

            if (next < first)
                hi = mid - 1;
            else if (next > last)
                lo = mid + 1;
            else
            {
                token_num = r->tokens[mid] + (next - first);
                bitsNeeded = TABLE_BITS - max + n;  /* to get TABLE_BITS bits into next */
                if (bitsNeeded > 0)
                {
                    r->next <<= bitsNeeded;
                    rest = MTX_BITIO_ReadValue(r->bio, bitsNeeded);
                    r->next |= rest;
                }
                else if (bitsNeeded < 0)
                {
                    int b = -bitsNeeded;
                    r->next >>= b;
                    r->bio->index += bitsNeeded ;
                }
                r->next &= NEXT_MASK;

                return token_num;
            }
        }
        /* This can't happen, we got a bad token number */

        STATE.error = ERR_bad_token_code;
        return 0;  /* to keep the compiler happy */
    }
}

/****************************************************************/
/* return the next token's string pointer and set its length */
FS_BYTE *ReadSymbol(_DS_ MTX_RA_DECOMP *t, FS_ULONG *len )
{
    int n;
    FS_BYTE *r;

    n = get_next_token(_PS_ t);
    *len = t->lengths[n];
    r = t->string + t->offsets[n];
    return r;
}
/****************************************************************/

#endif /* FS_ACT3 */
